@extends('layouts.app')

@section('title', 'Artworks')

@section('content')
<section class="py-5 bg-white">
    <div class="site-inner">

        <h2 class="text-center mb-4 fw-bold">
             Artworks
        </h2>

        {{-- FILTER BAR --}}
        <form id="filterForm" class="row g-3 align-items-end mb-4">

            {{-- PRICE FILTER --}}
           <div class="col-md-4 d-flex flex-column">
                <label class="fw-semibold mb-2 filter-label">
                    Price Range:
                    <span id="priceLabel">&#8377;0 - &#8377;50,00,000</span>
                </label>

                <div class="dual-range">
                    <div class="dual-range-track" id="priceTrack"></div>
                    <input type="range"
                           id="priceMinRange"
                           class="range-input range-input-min"
                           min="0"
                           max="5000000"
                           step="25000"
                           value="0">
                    <input type="range"
                           id="priceMaxRange"
                           class="range-input range-input-max"
                           min="0"
                           max="5000000"
                           step="25000"
                           value="5000000">
                </div>
                <input type="hidden" name="price_min" id="priceMin" value="0">
                <input type="hidden" name="price_max" id="priceMax" value="5000000">
            </div>

            {{-- FORM --}}
            <div class="col-md-4">
                <label class="fw-semibold">Form</label>
                <select name="form" class="form-select">
                    <option value="">All Forms</option>
                    @foreach($forms as $form)
                        <option value="{{ $form->id }}">{{ $form->name }}</option>
                    @endforeach
                </select>
            </div>

            {{-- THEMES --}}
            <div class="col-md-4">
                <label class="fw-semibold">Themes</label>
                <select id="themeDropdown" class="form-select">
                    <option value="">Select Theme</option>
                    @foreach($categories as $category)
                        <option value="{{ $category->id }}">{{ $category->name }}</option>
                    @endforeach
                </select>
            </div>

            {{-- RESET BUTTON --}}
<div class="col-md-12 text-end">
    <button type="button" id="resetFilters" class="btn btn-outline-dark px-4">
        Reset All Filters
    </button>
</div>

        </form>

        {{-- SELECTED THEMES --}}
        <div id="selectedThemes" class="d-flex flex-wrap gap-2 mb-4"></div>

        {{-- RESULTS --}}
        <div id="paintingResults">
            @include('front.paintings.partials.painting_grid')
        </div>

    </div>
</section>

@if(!empty($pageContents['artwork']->content))
<section class="seo-content">
    <div class="site-inner">
        {!! $pageContents['artwork']->content !!}
    </div>
</section>
@endif

@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function () {

    const baseUrl = "{{ \App\Models\SeoPage::url('artworks', '/artworks') }}";

    const form = document.getElementById('filterForm');
    const results = document.getElementById('paintingResults');

    const priceMinRange = document.getElementById('priceMinRange');
    const priceMaxRange = document.getElementById('priceMaxRange');
    const priceTrack = document.getElementById('priceTrack');
    const priceLabel = document.getElementById('priceLabel');
    const priceMin   = document.getElementById('priceMin');
    const priceMax   = document.getElementById('priceMax');

    const themeDropdown = document.getElementById('themeDropdown');
    const selectedBox   = document.getElementById('selectedThemes');
    const resetBtn      = document.getElementById('resetFilters');

    let selectedThemes = [];

    /* ================= PRICE SLIDER ================= */
    const PRICE_STEP = 25000;
    const PRICE_MAX_LIMIT = 5000000;

    function updatePriceUI(changedBy = 'max') {
        let minValue = Number(priceMinRange.value);
        let maxValue = Number(priceMaxRange.value);

        if ((maxValue - minValue) < PRICE_STEP) {
            if (changedBy === 'min') {
                minValue = maxValue - PRICE_STEP;
                priceMinRange.value = minValue;
            } else {
                maxValue = minValue + PRICE_STEP;
                priceMaxRange.value = maxValue;
            }
        }

        priceMin.value = minValue;
        priceMax.value = maxValue;
        priceLabel.innerText = `\u20B9${minValue.toLocaleString()} - \u20B9${maxValue.toLocaleString()}`;

        const minPercent = (minValue / PRICE_MAX_LIMIT) * 100;
        const maxPercent = (maxValue / PRICE_MAX_LIMIT) * 100;
        priceTrack.style.left = `${minPercent}%`;
        priceTrack.style.right = `${100 - maxPercent}%`;
    }

    priceMinRange.addEventListener('input', function () {
        updatePriceUI('min');
        fetchPaintings();
    });

    priceMaxRange.addEventListener('input', function () {
        updatePriceUI('max');
        fetchPaintings();
    });

    updatePriceUI();

    /* ================= THEME MULTI SELECT ================= */
    themeDropdown.addEventListener('change', function () {
        const id = this.value;
        const text = this.options[this.selectedIndex].text;

        if (!id || selectedThemes.includes(id)) return;

        selectedThemes.push(id);

        const chip = document.createElement('span');
        chip.className = 'badge bg-dark px-3 py-2';
        chip.innerHTML = `${text} ✕`;

        chip.onclick = () => {
            selectedThemes = selectedThemes.filter(t => t !== id);
            chip.remove();
            fetchPaintings();
        };

        selectedBox.appendChild(chip);
        fetchPaintings();
        this.value = '';
    });

    /* ================= RESET BUTTON ================= */
    resetBtn.addEventListener('click', function () {

        form.reset();

        priceMinRange.value = 0;
        priceMaxRange.value = 5000000;
        priceMin.value = 0;
        priceMax.value = 5000000;
        updatePriceUI();

        selectedThemes = [];
        selectedBox.innerHTML = '';

        fetchPaintings();
    });

    /* ================= AJAX FETCH ================= */
    function fetchPaintings(url = baseUrl) {

        const params = new URLSearchParams(new FormData(form));
        selectedThemes.forEach(id => params.append('categories[]', id));

        const urlObj = new URL(url, window.location.origin);
        const page = urlObj.searchParams.get('page');
        if (page) params.set('page', page);

        fetch(baseUrl + '?' + params.toString(), {
            headers: { 'X-Requested-With': 'XMLHttpRequest' }
        })
        .then(res => res.text())
        .then(html => {
            results.innerHTML = html;
        });
    }

    form.addEventListener('change', fetchPaintings);

    document.addEventListener('click', function (e) {
        const link = e.target.closest('.pagination a');
        if (!link) return;

        e.preventDefault();
        fetchPaintings(link.href);

        window.scrollTo({
            top: results.offsetTop - 80,
            behavior: 'smooth'
        });
    });

});
</script>

@endpush



<style>

#filterForm .fw-semibold{
    display:block;
    margin-bottom:10px;
}

.filter-label {
    line-height: 2.2;
    margin-bottom: 10px;
}

.dual-range {
    position: relative;
    height: 28px;
}

.dual-range::before {
    content: "";
    position: absolute;
    top: 12px;
    left: 0;
    right: 0;
    height: 4px;
    border-radius: 999px;
    background: #e3e3e3;
}

.dual-range-track {
    position: absolute;
    top: 12px;
    height: 4px;
    border-radius: 999px;
    background: #dc3545;
    left: 0;
    right: 0;
}

.range-input {
    position: absolute;
    left: 0;
    top: 0;
    width: 100%;
    height: 28px;
    margin: 0;
    pointer-events: none;
    background: transparent;
    -webkit-appearance: none;
    appearance: none;
}

.range-input::-webkit-slider-runnable-track {
    height: 4px;
    background: transparent;
}

.range-input::-moz-range-track {
    height: 4px;
    background: transparent;
}

.range-input::-webkit-slider-thumb {
    -webkit-appearance: none;
    appearance: none;
    width: 16px;
    height: 16px;
    margin-top: -6px;
    border-radius: 50%;
    border: 2px solid #fff;
    background: #dc3545;
    box-shadow: 0 1px 4px rgba(0,0,0,.25);
    pointer-events: auto;
    cursor: pointer;
}

.range-input::-moz-range-thumb {
    width: 16px;
    height: 16px;
    border-radius: 50%;
    border: 2px solid #fff;
    background: #dc3545;
    box-shadow: 0 1px 4px rgba(0,0,0,.25);
    pointer-events: auto;
    cursor: pointer;
}


.painting-card {
    border-radius: 10px;
    overflow: hidden;
    transition: .3s;
    background: #fff;
    height: 100%;
    cursor: pointer;
}

.painting-card:hover {
    transform: translateY(-5px);
}

.painting-image{
    aspect-ratio: 1 / 1;
    background: #f5f5f5;
    border-radius: 10px;

    display: flex;
    align-items: center;
    justify-content: center;
}


.painting-image img {
    width: 100%;
    height: 100%;
    object-fit: contain;
}

.painting-body {
    padding: 12px;
    text-align: center;
}

.price {
    font-size: 15px;
    font-weight: 600;
    color: #148c2e;
}

@media(max-width: 576px) {
    .painting-image { height: 180px; }
}

.pagination {
    justify-content: center;
    margin-top: 30px;
}
</style>

