@extends('layouts.app')

@section('title', $exhibition->title)

@section('content')

@php
    use Carbon\Carbon;

    $today = Carbon::today();
    $start = $exhibition->event_date ? Carbon::parse($exhibition->event_date) : null;
    $end   = $exhibition->end_date ? Carbon::parse($exhibition->end_date) : null;

    if ($start && $start->isFuture()) {
        $statusText = 'Upcoming Exhibition';
        $statusClass = 'badge-upcoming';
    } elseif ($end && $end->isPast()) {
        $statusText = 'Exhibition Ended';
        $statusClass = 'badge-ended';
    } else {
        $statusText = 'Ongoing Exhibition';
        $statusClass = 'badge-ongoing';
    }
@endphp

<!-- ================= TOP SECTION ================= -->
<section class="exhibition-top py-5 bg-white">
    <div class="container">

        <div class="top-grid">

            <!-- LEFT : COVER IMAGE -->
            <div class="top-image">
                <img src="{{ asset('storage/'.$exhibition->cover_image) }}"
                     alt="{{ $exhibition->title }}">
            </div>

            <!-- RIGHT : DETAILS -->
            <div class="top-content">

                <!-- <span class="event-badge {{ $statusClass }}">
                    {{ $statusText }}
                </span> -->

                <h1 class="event-title">{{ $exhibition->title }}</h1>

                <div class="event-meta">

                    @if($exhibition->event_date)
                        <p>📅
                            {{ Carbon::parse($exhibition->event_date)->format('F d, Y') }}
                            @if($exhibition->end_date)
                                – {{ Carbon::parse($exhibition->end_date)->format('F d, Y') }}
                            @endif
                        </p>
                    @endif

                    @if($exhibition->venue)
                        <p>📍 {{ $exhibition->venue }}</p>
                    @endif

                    <!-- @if($exhibition->event_time)
                        <p>⏰ {{ $exhibition->event_time }}</p>
                    @endif -->

                </div>

                @if($exhibition->description)
                    <p class="event-desc">
                        {{ $exhibition->description }}
                    </p>
                @endif

            </div>

        </div>

    </div>
</section>

<!-- ================= GALLERY ================= -->
@if($exhibition->images->count())
<section class="py-5 bg-white">
    <div class="container">

        <div class="gallery-grid">
            @foreach($exhibition->images as $index => $image)
                <div class="gallery-item">
                    <img
                        src="{{ asset('storage/'.$image->image) }}"
                        class="gallery-image"
                        data-index="{{ $index }}"
                        alt="{{ $exhibition->title }}"
                    >
                </div>
            @endforeach
        </div>

    </div>
</section>
@endif

<!-- ================= LIGHTBOX ================= -->
<div class="lightbox" id="lightbox">
    <span class="close-btn" id="closeLightbox">×</span>
    <span class="nav prev" id="prevSlide">❮</span>
    <img id="lightbox-img">
    <span class="nav next" id="nextSlide">❯</span>
</div>

{{-- SEO CONTENT (OPTIONAL – FROM ADMIN) --}}
@if(!empty($pageContents['exhibitions']->content))
<section class="seo-content py-5">
    <div class="container">
        {!! $pageContents['exhibitions']->content !!}
    </div>
</section>
@endif

@endsection

{{-- ================= STYLES ================= --}}
<style>
/* GRID */
.top-grid {
    display: grid;
    grid-template-columns: 1.2fr 1fr;
    gap: 60px;
}

/* IMAGE */
.top-image img {
    width: 100%;
    height: 460px;
    object-fit: cover;
    border-radius: 10px;
}

/* CONTENT */
.event-title {
    font-size: 34px;
    font-weight: 600;
    margin: 14px 0;
}

.event-meta {
    font-size: 15px;
    color: #555;
    margin-bottom: 18px;
}

.event-desc {
    font-size: 15px;
    color: #666;
    line-height: 1.7;
    max-width: 500px;
}

/* BADGE */
.event-badge {
    display: inline-block;
    padding: 6px 14px;
    font-size: 13px;
    font-weight: 600;
    border-radius: 30px;
}

.badge-upcoming { background: #f0ad4e; color: #fff; }
.badge-ongoing  { background: #5cb85c; color: #fff; }
.badge-ended    { background: #777; color: #fff; }

/* GALLERY */
.gallery-grid {
    display: grid;
    grid-template-columns: repeat(5, 1fr);
    gap: 22px;
}

.gallery-item {
    height: 190px;
    overflow: hidden;
    border-radius: 8px;
}

.gallery-item img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    cursor: pointer;
    transition: transform .4s ease;
}

.gallery-item img:hover {
    transform: scale(1.08);
}

/* LIGHTBOX */
.lightbox {
    position: fixed;
    inset: 0;
    background: rgba(0,0,0,0.95);
    display: none;
    align-items: center;
    justify-content: center;
    z-index: 99999;
}

.lightbox img {
    width: 80vw;
    height: 80vh;
    object-fit: contain;
}

.close-btn {
    position: absolute;
    top: 20px;
    right: 30px;
    font-size: 40px;
    color: #fff;
    cursor: pointer;
}

.nav {
    position: absolute;
    top: 50%;
    font-size: 50px;
    color: #fff;
    cursor: pointer;
}

.prev { left: 40px; }
.next { right: 40px; }

/* RESPONSIVE */
@media (max-width: 992px) {
    .top-grid { grid-template-columns: 1fr; }
    .gallery-grid { grid-template-columns: repeat(3, 1fr); }
}

@media (max-width: 576px) {
    .gallery-grid { grid-template-columns: repeat(2, 1fr); }
}
</style>

{{-- ================= SCRIPT ================= --}}
<script>
document.addEventListener("DOMContentLoaded", () => {

    const images = document.querySelectorAll('.gallery-image');
    const lightbox = document.getElementById('lightbox');
    const lightboxImg = document.getElementById('lightbox-img');

    let currentIndex = 0;

    images.forEach((img, i) => {
        img.addEventListener('click', () => {
            currentIndex = i;
            lightboxImg.src = img.src;
            lightbox.style.display = 'flex';
        });
    });

    document.getElementById('closeLightbox').onclick = () => {
        lightbox.style.display = 'none';
    };

    document.getElementById('nextSlide').onclick = () => slide(1);
    document.getElementById('prevSlide').onclick = () => slide(-1);

    function slide(step) {
        currentIndex = (currentIndex + step + images.length) % images.length;
        lightboxImg.src = images[currentIndex].src;
    }

    lightbox.addEventListener('click', e => {
        if (e.target === lightbox) {
            lightbox.style.display = 'none';
        }
    });

    document.addEventListener('keydown', e => {
        if (e.key === 'Escape') lightbox.style.display = 'none';
        if (e.key === 'ArrowRight') slide(1);
        if (e.key === 'ArrowLeft') slide(-1);
    });

});
</script>
