@extends('admin.layout')

@section('title', 'Themes')

@section('content')
<style>
/* Premium theme management UI (admin-only visual layer). */
.theme-admin-shell {
    --theme-accent: #e6732c;
    --theme-accent-dark: #cf5f1d;
    --theme-border: #e8edf4;
    --theme-text: #1f2937;
    --theme-muted: #6b7280;
}

.theme-hero {
    border-radius: 18px;
    border: 1px solid #f4d5bf;
    background: linear-gradient(130deg, #fff8f2 0%, #ffffff 58%, #fff1e6 100%);
    box-shadow: 0 16px 38px rgba(15, 23, 42, 0.08);
    padding: 18px 20px;
}

.theme-kicker {
    margin: 0 0 4px 0;
    text-transform: uppercase;
    letter-spacing: .12em;
    font-size: .74rem;
    font-weight: 700;
    color: var(--theme-accent);
}

.theme-title {
    margin: 0;
    color: var(--theme-text);
    font-weight: 700;
}

.theme-subtitle {
    margin: 6px 0 0;
    color: var(--theme-muted);
    font-size: .95rem;
}

.theme-btn-primary {
    --bs-btn-color: #ffffff;
    --bs-btn-hover-color: #ffffff;
    --bs-btn-active-color: #ffffff;
    border: none;
    border-radius: 12px;
    background-color: #e6732c !important;
    background-image: linear-gradient(135deg, #e6732c 0%, #f09552 100%);
    color: #fff !important;
    font-weight: 600;
    line-height: 1.2;
    box-shadow: 0 14px 24px rgba(230, 115, 44, 0.22);
}

.theme-btn-primary:hover {
    background-color: #cf5f1d !important;
    background-image: linear-gradient(135deg, #cf5f1d 0%, #e6732c 100%);
    color: #fff !important;
}

.theme-card {
    border-radius: 18px;
    border: 1px solid var(--theme-border);
    box-shadow: 0 24px 52px rgba(15, 23, 42, 0.06);
    overflow: hidden;
}

.theme-table {
    margin-bottom: 0;
}

.theme-table thead th {
    border-bottom: 1px solid #e9edf5;
    background: #f8fafc;
    color: #3a475a;
    font-size: .78rem;
    letter-spacing: .08em;
    text-transform: uppercase;
    font-weight: 700;
    white-space: nowrap;
}

.theme-table tbody td {
    border-color: #f2f4f8;
    vertical-align: middle;
}

.theme-name {
    font-weight: 600;
    color: var(--theme-text);
}

.theme-image {
    width: 52px;
    height: 52px;
    border-radius: 10px;
    object-fit: cover;
    border: 1px solid #e5e7eb;
    background: #fff;
}

.theme-slug {
    display: inline-block;
    border: 1px solid #e5eaf1;
    background: #f8fafc;
    color: #475569;
    border-radius: 999px;
    font-size: .78rem;
    padding: 4px 10px;
}

.theme-status {
    display: inline-block;
    font-size: .74rem;
    font-weight: 700;
    border-radius: 999px;
    padding: 4px 9px;
}

.theme-status.active {
    background: #dcfce7;
    color: #166534;
}

.theme-status.inactive {
    background: #eceff3;
    color: #475569;
}

.theme-action {
    border-radius: 10px;
    font-weight: 600;
    min-width: 70px;
}

.theme-empty {
    color: #6b7280;
    text-align: center;
    padding: 42px 12px;
}

.theme-modal .modal-content {
    border-radius: 16px;
    border: 1px solid #e7ebf3;
    box-shadow: 0 20px 42px rgba(15, 23, 42, 0.14);
}

.theme-modal .modal-header {
    background: #fbfcff;
    border-bottom: 1px solid #edf1f7;
}

.theme-modal .modal-title {
    font-weight: 700;
    color: #273244;
}

.theme-modal .form-label {
    font-size: .78rem;
    letter-spacing: .06em;
    text-transform: uppercase;
    font-weight: 700;
    color: #4b5a72;
    margin-bottom: .4rem;
}

.theme-modal .form-control,
.theme-modal .form-select {
    min-height: 44px;
    border-radius: 12px;
    border: 1px solid #d9e0eb;
}

.theme-modal .form-control:focus,
.theme-modal .form-select:focus {
    border-color: #ebb387;
    box-shadow: 0 0 0 .25rem rgba(230, 115, 44, 0.12);
}

.theme-modal .help-note {
    color: #6b7280;
    font-size: .78rem;
}

@media (max-width: 992px) {
    .theme-title {
        font-size: 1.3rem;
    }
}
</style>

<div class="theme-admin-shell container-fluid py-4">
    <div class="theme-hero mb-4 d-flex flex-column flex-lg-row align-items-lg-center justify-content-between gap-3">
        <div>
            <p class="theme-kicker">Category Settings</p>
            <h2 class="theme-title">Themes</h2>
            <p class="theme-subtitle">Create and manage frontend themes from one place.</p>
        </div>

        <button class="btn theme-btn-primary px-4" data-bs-toggle="modal" data-bs-target="#createCategoryModal">
            Add Theme
        </button>
    </div>

    @if(session('success'))
        <div class="alert alert-success border-0 shadow-sm">{{ session('success') }}</div>
    @endif

    @if(session('error'))
        <div class="alert alert-danger border-0 shadow-sm">{{ session('error') }}</div>
    @endif

    @if($errors->any())
        <div class="alert alert-danger border-0 shadow-sm">
            <strong>Please fix the following:</strong>
            <ul class="mb-0 mt-2">
                @foreach($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <div class="theme-card bg-white">
        <div class="table-responsive">
            <table class="table table-hover align-middle theme-table">
                <thead>
                    <tr>
                        <th class="ps-3">Sequence</th>
                        <th>Name</th>
                        <th>Image</th>
                        <th>Slug</th>
                        <th>Status</th>
                        <th class="text-end pe-3" style="width: 130px;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($categories as $category)
                        <tr>
                            <td class="ps-3">{{ $loop->iteration }}</td>
                            <td><span class="theme-name">{{ $category->name }}</span></td>
                            <td>
                                @if($category->image)
                                    <img src="{{ asset('uploads/categories/' . $category->image) }}" class="theme-image" alt="{{ $category->name }}">
                                @else
                                    <span class="text-muted">No image</span>
                                @endif
                            </td>
                            <td><span class="theme-slug">{{ $category->slug ?: '-' }}</span></td>
                            <td>
                                <span class="theme-status {{ $category->status === 'active' ? 'active' : 'inactive' }}">
                                    {{ ucfirst($category->status) }}
                                </span>
                            </td>
                            <td class="text-end pe-3">
                                <button
                                    class="btn btn-sm btn-outline-primary theme-action"
                                    data-bs-toggle="modal"
                                    data-bs-target="#editCategoryModal"
                                    data-id="{{ $category->id }}"
                                    data-name="{{ $category->name }}"
                                    data-slug="{{ $category->slug }}"
                                    data-image="{{ $category->image }}"
                                    data-status="{{ $category->status }}">
                                    Edit
                                </button>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" class="theme-empty">No themes found.</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>

    <div class="mt-3">
        {{ $categories->links() }}
    </div>
</div>

{{-- Create Theme modal --}}
<div class="modal fade theme-modal" id="createCategoryModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create Theme</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>

            <form action="{{ route('admin.categories.store') }}" method="POST" enctype="multipart/form-data">
                @csrf
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Name</label>
                        <input type="text" name="name" class="form-control" placeholder="Enter theme name" required>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Theme Image</label>
                        <input type="file" name="image" class="form-control" accept="image/*">
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Slug</label>
                        <input type="text" name="slug" class="form-control" placeholder="Auto-generated if left blank">
                    </div>

                    <p class="help-note mb-0">Note: Name is required. Slug can be manual or auto-generated.</p>
                </div>

                <div class="modal-footer">
                    <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn theme-btn-primary px-4">Save</button>
                </div>
            </form>
        </div>
    </div>
</div>

{{-- Edit Theme modal --}}
<div class="modal fade theme-modal" id="editCategoryModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Theme</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>

            <form id="editCategoryForm" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')

                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Name</label>
                        <input type="text" id="edit_name" name="name" class="form-control" required>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">New Image</label>
                        <input type="file" name="image" class="form-control" accept="image/*">
                        <div id="editImagePreview" class="mt-2"></div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Slug</label>
                        <input type="text" id="edit_slug" name="slug" class="form-control">
                    </div>

                    <div>
                        <label class="form-label">Status</label>
                        <select id="edit_status" name="status" class="form-select">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </div>

                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-danger me-auto" onclick="submitCategoryDelete()">Delete</button>
                    <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn theme-btn-primary px-4">Update</button>
                </div>
            </form>

            <form id="deleteCategoryForm" method="POST" style="display:none;">
                @csrf
                @method('DELETE')
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
// Edit modal hydrates selected row data and wires update/delete actions.
const editModal = document.getElementById('editCategoryModal');
if (editModal) {
    editModal.addEventListener('show.bs.modal', function (event) {
        const button = event.relatedTarget;
        const id = button.getAttribute('data-id');

        document.getElementById('editCategoryForm').action = `/admin/categories/${id}`;
        document.getElementById('deleteCategoryForm').action = `/admin/categories/${id}`;

        document.getElementById('edit_name').value = button.getAttribute('data-name') || '';
        document.getElementById('edit_slug').value = button.getAttribute('data-slug') || '';

        const statusEl = document.getElementById('edit_status');
        if (statusEl) {
            statusEl.value = button.getAttribute('data-status') || 'active';
        }

        const image = button.getAttribute('data-image');
        // Use app URL base instead of hardcoded root path for cPanel/subfolder deployments.
        const categoryBaseUrl = @json(asset('uploads/categories'));
        document.getElementById('editImagePreview').innerHTML = image
            ? `<img src="${categoryBaseUrl}/${image}" width="120" class="border rounded" alt="Theme">`
            : '<small class="text-muted">No image uploaded</small>';
    });
}

function submitCategoryDelete() {
    if (confirm('Delete this theme?')) {
        document.getElementById('deleteCategoryForm').submit();
    }
}
</script>
@endpush
