function asStringAttributes(object) {
  return Object.entries(object).map(([key, value]) => ({
    key,
    value: {
      stringValue: String(value)
    }
  }));
}
function asTypedAttributes(object) {
  return Object.entries(object).map(([key, value]) => ({
    key,
    value: asTypedValue(value)
  }));
}
function asTypedValue(value) {
  if (Array.isArray(value)) {
    return {
      arrayValue: {
        values: value.map(val => asTypedValue(val))
      }
    };
  }
  switch (typeof value) {
    case 'boolean':
      return {
        boolValue: Boolean(value)
      };
    // Javascript unfortunately has only the `number` type (ignoring
    // BigInt, which I don't think we support on the backend yet).
    // The `number` type is actually a float; and while we can test
    // for whether or not the `number` is actually representing an
    // "integer", the MDN docs imply that even that is not really a
    // reliable test. So - we just tell the backend that everything
    // is a float. That's technically the truth!
    //
    // Now - there are some consequences: primarily, because we'll
    // then store everything in a Float64 column in Clickhouse, it
    // could result in accumulated inaccuracies particularly when
    // aggregating over a query result. But it also avoids another
    // particularly nasty edge case - if we send a mixed array of
    // intValue/doubleValue, our backend will pick whichever type
    // it sees first (a reasonable choice, I think) and then coerce
    // from there - however, that could mean we coerce a lot of
    // floats to ints and lose much more precision than in the
    // aggregation edge case.
    //
    // And so: we tell the backend that everything is a float.
    case 'number':
      return {
        doubleValue: Number(value)
      };
    case 'string':
    default:
      return {
        stringValue: String(value)
      };
  }
}

export { asStringAttributes, asTypedAttributes };
