import { defineProperty as _defineProperty, objectSpread2 as _objectSpread2 } from '../_virtual/_rollupPluginBabelHelpers.mjs';
import { isCompositeMonorailEvent } from '../events/events.mjs';

class ConsentTrackingProvider {
  constructor(config) {
    this.version = config.version;
  }
}
class ConsentTrackingMiddleware {
  constructor(config, isStrictlyNecessary = () => false) {
    _defineProperty(this, "eventsAwaitingConsent", []);
    // Check if provider exists
    if (!(config !== null && config !== void 0 && config.provider)) {
      throw new MonorailUnableToInstantiateMiddlewareError('ConsentTrackingMiddleware requires an instance of ConsentTrackingProvider');
    }
    this.isStrictlyNecessary = isStrictlyNecessary;
    this.provider = config.provider;
  }
  async do(event, proceed) {
    if (isCompositeMonorailEvent(event)) {
      // Single event
      const eventWithConsent = await this.provider.annotateEvent(event);
      if (this.isConsentGivenForEmission(eventWithConsent)) {
        // When consent is granted, first process any events that were previously
        // buffered while waiting for consent, then proceed with the current event
        await this.processBufferedEvents(proceed);
        return proceed(eventWithConsent);
      }

      // If no consent is given, check if event is strictly necessary
      if (this.isStrictlyNecessary(eventWithConsent)) {
        return proceed(eventWithConsent);
      }

      // If no consent is given and event is not strictly necessary, buffer the event
      // and return a response indicating the event wasn't sent.
      this.eventsAwaitingConsent.push(event);
      return Promise.resolve({
        status: 0,
        message: 'Consent not granted and event not marked strictly necessary, event not sent'
      });
    } else {
      // Batch events
      const firstEventHasConsent = this.isConsentGivenForEmission(await this.provider.annotateEvent(event.events[0]));

      // If the first event in the batch has consent, it indicates the user has
      // granted consent.
      if (firstEventHasConsent) {
        // At this point, we can process all previously buffered
        // events that were awaiting consent.
        await this.processBufferedEvents(proceed);

        // Annotate and process the current batch events.
        const annotatedBatchedEvent = await Promise.all(event.events.map(event => this.provider.annotateEvent(event)));
        return proceed(_objectSpread2(_objectSpread2({}, event), {}, {
          events: annotatedBatchedEvent
        }));
      }

      // If the first event in the batch does not have consent, we need to
      // filter out events that are strictly necessary and buffer the rest.
      const strictlyNecessaryEvents = event.events.filter(event => {
        if (this.isStrictlyNecessary(event)) {
          return true;
        }
        this.eventsAwaitingConsent.push(event);
        return false;
      });

      // Process any neccessary events as the batch.
      if (strictlyNecessaryEvents.length > 0) {
        const annotatedEvents = await Promise.all(strictlyNecessaryEvents.map(event => this.provider.annotateEvent(event)));
        return proceed(_objectSpread2(_objectSpread2({}, event), {}, {
          events: annotatedEvents
        }));
      }

      // If no events are strictly necessary, we can return an error
      return Promise.resolve({
        status: 0,
        message: 'Consent not granted for any event, and no event marked strictly necessary, event batch not sent'
      });
    }
  }
  isConsentGivenForEmission(event) {
    var _event$metadata;
    const consent = (_event$metadata = event.metadata) === null || _event$metadata === void 0 ? void 0 : _event$metadata.consent;
    const requiredConsentTypes = this.provider.getRequiredConsentForEmission();
    return Boolean(Array.isArray(consent) && consent.some(consent => requiredConsentTypes.includes(consent)));
  }
  async processBufferedEvents(proceed) {
    if (this.eventsAwaitingConsent.length === 0) {
      return;
    }
    const events = this.eventsAwaitingConsent;
    this.eventsAwaitingConsent = [];
    const annotatedBatchedEvents = await Promise.all(events.map(event => this.provider.annotateEvent(event)));
    await proceed({
      events: annotatedBatchedEvents
    });
  }
}
class MonorailUnableToInstantiateMiddlewareError extends Error {
  constructor(message) {
    super(message);
    Object.setPrototypeOf(this, MonorailUnableToInstantiateMiddlewareError.prototype);
  }
}

export { ConsentTrackingMiddleware, ConsentTrackingProvider, MonorailUnableToInstantiateMiddlewareError };
