import { isTestEnv } from '../utils/logger.mjs';

const CONSENT_COOKIE_NAME = "_tracking_consent";

function safeDecodeURIComponent(str) {
  try {
    return decodeURIComponent(str);
  } catch (e) {
    return "";
  }
}

function canAccessCookies() {
  // Some iframes throw a SecurityError when accessing cookies. In those cases
  // we'll fall back to assume there is no tracking consent cookie. If they
  // can't access the cookies, they don't have any unique identifiers anyway.
  //
  // In any case, avoid an uncaught error and the client stopping logic abruptly.
  try {
    return document.cookie;
  } catch {
    return false;
  }
} // TODO: Due to some limitations and unexpected behaviour of systems that use our API,
// the "fallbackRequest" is disabled by default. We should revisit this in the future.


function fetchCookieFromCookies(cookieName, fallbackRequest = false) {
  const cookies = canAccessCookies() ? document.cookie.split("; ") : [];

  for (let i = 0; i < cookies.length; i++) {
    const [name, value] = cookies[i].split("=");

    if (cookieName === safeDecodeURIComponent(name)) {
      const cookieValue = safeDecodeURIComponent(value);
      return cookieValue;
    }
  } // If the fetched cookie is '_tracking_consent' and it does not exist we want to make a synchronous request that will
  // trigger is generation. There are rare cases in the checkout page that, due to a race condition, buyers
  // are being tracked but shouldn't be, because the cookie is not present. This is a workaround to fix that.


  if (fallbackRequest && cookieName === "_tracking_consent" && !window.localStorage.getItem("tracking_consent_fetched")) {
    // This code should not run in test server side. This avoids blocking the event loop.
    if (isTestEnv()) {
      return undefined;
    } // Note: This only works for liquid shops or checkout.


    console.debug("_tracking_consent missing");
    makeSynchronousRequest(); // We want to make the request only once per page load.

    window.localStorage.setItem("tracking_consent_fetched", "true");
    return fetchCookieFromCookies(cookieName, false);
  }
}

function makeSynchronousRequest(url = "/") {
  // Make a HEAD request to root domain to trigger the browser to update the _tracking_consent cookie.
  const request = new XMLHttpRequest(); // Setting the third parameter to false makes the request synchronous

  request.open("HEAD", url, false);
  request.withCredentials = true;
  request.send();
}
function readCookie() {
  return fetchCookieFromCookies(CONSENT_COOKIE_NAME);
}

function isUriEncoded(str) {
  return str === encodeURIComponent(safeDecodeURIComponent(str));
}

function writeCookie(cookieName, rootDomain, maxAge, cookieValue) {
  if (!isUriEncoded(cookieValue)) throw new TypeError("Cookie value is not correctly URI encoded.");
  if (!isUriEncoded(cookieName)) throw new TypeError("Cookie name is not correctly URI encoded.");
  let cookie = `${cookieName}=${cookieValue}`;
  cookie += "; path=/";

  if (rootDomain) {
    cookie += `; domain=${rootDomain}`;
  }

  cookie += `; expires=${new Date(new Date().getTime() + maxAge).toUTCString()}`; // Don't set the cookie as secure, because the backend doesn't. We want to
  // match the backend.

  document.cookie = cookie;
}

export { CONSENT_COOKIE_NAME, makeSynchronousRequest, readCookie, writeCookie };
