import { ConsentTrackingProvider } from '@shopify/monorail/lib/middleware/consent-tracking-middleware';
import { analyticsProcessingAllowed, marketingAllowed, saleOfDataAllowed, preferencesProcessingAllowed } from './consent-api.mjs';

function getConsentV1Data() {
  const consent = [];

  if (analyticsProcessingAllowed()) {
    consent.push("analytics");
  }

  if (marketingAllowed()) {
    consent.push("marketing");
  }

  if (saleOfDataAllowed()) {
    consent.push("sale_of_data");
  }

  if (preferencesProcessingAllowed()) {
    consent.push("preferences");
  }

  return consent;
}
/**
 * The annotate function should be exported separately from the class
 * to allow users who do not depend on the Monorail class to still access and use it.
 */


function annotateEvent(event, version) {
  switch (version) {
    case "v1":
      {
        const consentData = getConsentV1Data();
        return { ...event,
          metadata: { ...(event === null || event === void 0 ? void 0 : event.metadata),
            consent: consentData,

            /** To be returned by the consent-tracking-api package */
            consent_provider: "consent-tracking-api",
            consent_version: version
          }
        };
      }

    default:
      throw new MonorailConsentTrackingApiProviderVersionError(version || "unknown");
  }
}
class ConsentTrackingApiProvider extends ConsentTrackingProvider {
  async annotateEvent(event) {
    return Promise.resolve(annotateEvent(event, this.version));
  }

  getRequiredConsentForEmission() {
    switch (this.version) {
      case "v1":
        return ["analytics", "marketing"];

      default:
        throw new MonorailConsentTrackingApiProviderVersionError(this.version || "unknown");
    }
  }

}
class MonorailConsentTrackingApiProviderVersionError extends Error {
  constructor(version) {
    super(`Version ${version} is not supported by the consent-tracking-api provider`);
    this.name = "MonorailConsentTrackingApiProviderVersionError";
    Object.setPrototypeOf(this, MonorailConsentTrackingApiProviderVersionError.prototype);
  }

}

export { ConsentTrackingApiProvider, MonorailConsentTrackingApiProviderVersionError, annotateEvent };
