import { DisplayConsentValues, ConsentKeys, ConsentValues, ConsentDisplayKeys, StorefrontApiConsentMetadata } from './types/index.mjs';
import { validStorageDoesNotExist, getCMPConsentValues, getRegionValue, getCustomValue, getCustomValues, transformKeyToDisplayName, transformConsentToDisplaySchema, getConsentId, marketingProcessingAllowed, analyticsProcessingAllowed, preferencesProcessingAllowed, saleOfDataAllowed, saleOfDataRegion, shouldShowBanner, getCCPAConsentValue } from './consent-storage.mjs';
export { saleOfDataRegion } from './consent-storage.mjs';
import { apiVersion } from './api-version.mjs';
import { storefrontApiUpdateTrackingConsent } from './storefront-api-client.mjs';
import { Monorail } from './monorail.mjs';

function validateConsent(consent) {
  if (typeof consent !== "boolean" && typeof consent !== "object") {
    throw TypeError(`setTrackingConsent must be called with a boolean or object consent value`);
  }

  if (typeof consent === "object") {
    const submittedConsentKeys = Object.keys(consent);

    if (submittedConsentKeys.length === 0) {
      throw TypeError("The submitted consent object is empty.");
    }

    const validConsentKeys = [ConsentDisplayKeys.MARKETING, ConsentDisplayKeys.ANALYTICS, ConsentDisplayKeys.PREFERENCES, ConsentDisplayKeys.SALE_OF_DATA, ConsentDisplayKeys.EMAIL, // Storefront Api consent can give some extra metadata
    StorefrontApiConsentMetadata.ROOT_DOMAIN, StorefrontApiConsentMetadata.CHECKOUT_ROOT_DOMAIN, StorefrontApiConsentMetadata.STOREFRONT_ROOT_DOMAIN, StorefrontApiConsentMetadata.STOREFRONT_ACCESS_TOKEN, StorefrontApiConsentMetadata.HEADLESS_STOREFRONT, StorefrontApiConsentMetadata.IS_EXTENSION_TOKEN, StorefrontApiConsentMetadata.METAFIELDS];

    for (const purposeKey of submittedConsentKeys) {
      if (!validConsentKeys.includes(purposeKey)) {
        throw TypeError(`The submitted consent object should only contain the following keys: ${validConsentKeys.join(", ")}. Extraneous key: ${purposeKey}.`);
      }
    }
  }
}

function setTrackingConsent(consent, callback) {
  const monorail = new Monorail();
  monorail.produce("setTrackingConsent", apiVersion);
  validateConsent(consent);

  if (callback !== undefined && typeof callback !== "function") {
    throw TypeError("setTrackingConsent must be called with a callback function if the callback argument is provided");
  }

  const referrerData = getReferrerData(consent.analytics);
  const landingPageData = getLandingPageData(consent.analytics);
  return storefrontApiUpdateTrackingConsent(currentVisitorConsent, {
    granular_consent: consent,
    ...(referrerData !== null && {
      referrer: referrerData
    }),
    ...(landingPageData !== null && {
      landing_page: landingPageData
    })
  }, callback);
}

function getConsentDisplayValue(consentValue) {
  if (validStorageDoesNotExist()) {
    return DisplayConsentValues.NO_VALUE;
  }

  if (consentValue === ConsentValues.NO_VALUE) {
    return DisplayConsentValues.NO_INTERACTION;
  }

  return transformConsentToDisplaySchema(consentValue);
}

function getTrackingConsent() {
  const monorail = new Monorail();
  monorail.produce("getTrackingConsent", apiVersion);

  if (validStorageDoesNotExist()) {
    return DisplayConsentValues.NO_VALUE;
  }

  const consentValue = getCMPConsentValues();

  if (consentValue[ConsentKeys.MARKETING] === ConsentValues.ACCEPTED && consentValue[ConsentKeys.ANALYTICS] === ConsentValues.ACCEPTED) {
    return DisplayConsentValues.ACCEPTED;
  }

  if (consentValue[ConsentKeys.MARKETING] === ConsentValues.DECLINED || consentValue[ConsentKeys.ANALYTICS] === ConsentValues.DECLINED) {
    return DisplayConsentValues.DECLINED;
  }

  return DisplayConsentValues.NO_INTERACTION;
}

function getReferrerData(consent) {
  if (!consent) {
    return null;
  }

  return isBuyerOnFirstPageOfVisit() ? document.referrer : "";
}

function getLandingPageData(consent) {
  if (!consent) {
    return null;
  }

  return isBuyerOnFirstPageOfVisit() ? window.location.pathname + window.location.search : "/";
}
function getRegion() {
  return getRegionValue();
}
function isBuyerOnFirstPageOfVisit() {
  if (document.referrer === "") {
    return true;
  }

  const a = document.createElement("a");
  a.href = document.referrer;
  return window.location.hostname != a.hostname;
}
function hasConsentForGDPR() {
  return marketingProcessingAllowed() && analyticsProcessingAllowed();
}
function userCanBeTracked() {
  if (validStorageDoesNotExist()) {
    return true;
  }

  return hasConsentForGDPR();
} // true if Sale of Data has been accepted,
// false if it has been declined,
// null if there is no signal or we shouldn't read it.

function GPCSignal() {
  if (!saleOfDataRegion()) {
    return null;
  } // GPC seems to only be readable in firefox and brave.
  // Follows inverse logic. Positive signal means no consent.


  if (typeof navigator.globalPrivacyControl === "string") {
    // firefox
    return navigator.globalPrivacyControl === "1" ? false : true;
  }

  if (typeof navigator.globalPrivacyControl === "boolean") {
    // brave
    return !navigator.globalPrivacyControl;
  }

  return null;
}
function shouldShowGDPRBanner() {
  return shouldShowBanner() && getTrackingConsent() === DisplayConsentValues.NO_INTERACTION;
}
function getCCPAConsent() {
  if (GPCSignal() === false) return DisplayConsentValues.DECLINED;
  return getConsentDisplayValue(getCCPAConsentValue());
}
function doesMerchantSupportGranularConsent() {
  return true;
}
function haveAnalyticsConsent() {
  return analyticsProcessingAllowed();
}
function havePreferencesConsent() {
  return preferencesProcessingAllowed();
}
function haveMarketingConsent() {
  return marketingProcessingAllowed();
}
function haveSaleOfDataConsent() {
  return saleOfDataAllowed();
}
function getTrackingConsentMetafield(key) {
  return getCustomValue(key);
}
function getTrackingConsentMetafields() {
  return getCustomValues();
}
function currentVisitorConsent() {
  const currentConsent = {};
  const consent = getCMPConsentValues();

  for (const purpose of Object.keys(consent)) {
    currentConsent[transformKeyToDisplayName(purpose)] = transformConsentToDisplaySchema(consent[purpose]);
  }

  return currentConsent;
}
function consentId() {
  return getConsentId();
}

export { GPCSignal, consentId, currentVisitorConsent, doesMerchantSupportGranularConsent, getCCPAConsent, getLandingPageData, getRegion, getTrackingConsent, getTrackingConsentMetafield, getTrackingConsentMetafields, hasConsentForGDPR, haveAnalyticsConsent, haveMarketingConsent, havePreferencesConsent, haveSaleOfDataConsent, isBuyerOnFirstPageOfVisit, setTrackingConsent, shouldShowGDPRBanner, userCanBeTracked };
