<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

return new class extends Migration
{
    public function up(): void
    {
        // Recover corrupted id columns where AUTO_INCREMENT/PRIMARY KEY metadata was lost.
        $this->repairPrimaryAutoIncrementIds();

        // Recreate missing feature tables required by controllers and shared view data.
        $this->ensurePaintingsTable();
        $this->ensureSculpturesTable();
        $this->ensurePopularSearchesTable();
        $this->ensureTestimonialsTable();
        $this->ensurePageContentsTable();

        // Keep many-to-many theme mapping stable for artwork/category flows.
        $this->ensureCategoryPaintingTable();

        // Align SEO table columns with current SEO controllers and slug routing logic.
        $this->repairSeoPagesTable();
        $this->seedMissingSeoPages();
    }

    public function down(): void
    {
        // Recovery migration is intentionally non-destructive in rollback.
    }

    private function repairPrimaryAutoIncrementIds(): void
    {
        $tables = [
            'admins',
            'admin_users',
            'artists',
            'artwork_types',
            'art_advisories',
            'blogs',
            'blog_categories',
            'categories',
            'category_painting',
            'contact_enquiries',
            'contact_settings',
            'events',
            'exclusive_sections',
            'exhibitions',
            'exhibition_images',
            'failed_jobs',
            'favourites',
            'footer_settings',
            'forms',
            'galleries',
            'hero_sliders',
            'jobs',
            'job_batches',
            'media_presences',
            'mediums',
            'about_sections',
            'about_creativity_sections',
            'seo_pages',
            'seo_page_slug_histories',
            'users',
        ];

        foreach ($tables as $tableName) {
            if (!Schema::hasTable($tableName) || !Schema::hasColumn($tableName, 'id')) {
                continue;
            }

            $idMeta = DB::table('information_schema.columns')
                ->select('data_type', 'column_type', 'column_key', 'extra')
                ->where('table_schema', DB::raw('DATABASE()'))
                ->where('table_name', $tableName)
                ->where('column_name', 'id')
                ->first();

            if (!$idMeta) {
                continue;
            }

            $isNumericId = in_array(strtolower((string) $idMeta->data_type), ['bigint', 'int', 'mediumint', 'smallint', 'tinyint'], true);
            if (!$isNumericId) {
                continue;
            }

            $hasPrimaryOnId = DB::table('information_schema.statistics')
                ->where('table_schema', DB::raw('DATABASE()'))
                ->where('table_name', $tableName)
                ->where('index_name', 'PRIMARY')
                ->where('column_name', 'id')
                ->exists();

            if (!$hasPrimaryOnId) {
                Schema::table($tableName, function (Blueprint $table) {
                    $table->primary('id');
                });
            }

            $isAutoIncrement = str_contains(strtolower((string) $idMeta->extra), 'auto_increment');
            if (!$isAutoIncrement) {
                Schema::table($tableName, function (Blueprint $table) {
                    $table->unsignedBigInteger('id', true)->change();
                });
            }
        }
    }

    private function ensurePaintingsTable(): void
    {
        if (!Schema::hasTable('paintings')) {
            Schema::create('paintings', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('artwork_type_id')->nullable();
                $table->string('painting_code', 10)->nullable();
                $table->string('slug', 50)->nullable();
                $table->string('title');
                $table->unsignedBigInteger('artist_id')->nullable();
                $table->unsignedBigInteger('category_id')->nullable();
                $table->unsignedBigInteger('gallery_id')->nullable();
                $table->unsignedBigInteger('medium_id')->nullable();
                $table->unsignedBigInteger('form_id')->nullable();
                $table->text('description')->nullable();
                $table->decimal('price', 12, 2)->nullable();
                $table->decimal('height', 8, 2)->nullable();
                $table->decimal('width', 8, 2)->nullable();
                $table->boolean('price_on_request')->default(false);
                $table->string('image')->nullable();
                $table->enum('status', ['active', 'inactive', 'sold'])->default('active');
                $table->string('updated_by', 100)->nullable();
                $table->timestamps();

                $table->unique('painting_code');
                $table->index('slug');
            });
        }

        Schema::table('paintings', function (Blueprint $table) {
            if (!Schema::hasColumn('paintings', 'painting_code')) {
                $table->string('painting_code', 10)->nullable()->after('id');
            }
            if (!Schema::hasColumn('paintings', 'slug')) {
                $table->string('slug', 50)->nullable()->after('painting_code');
            }
            if (!Schema::hasColumn('paintings', 'artist_id')) {
                $table->unsignedBigInteger('artist_id')->nullable()->after('title');
            }
            if (!Schema::hasColumn('paintings', 'gallery_id')) {
                $table->unsignedBigInteger('gallery_id')->nullable()->after('category_id');
            }
            if (!Schema::hasColumn('paintings', 'artwork_type_id')) {
                $table->unsignedBigInteger('artwork_type_id')->nullable()->after('id');
            }
            if (!Schema::hasColumn('paintings', 'medium_id')) {
                $table->unsignedBigInteger('medium_id')->nullable()->after('painting_code');
            }
            if (!Schema::hasColumn('paintings', 'form_id')) {
                $table->unsignedBigInteger('form_id')->nullable()->after('medium_id');
            }
            if (!Schema::hasColumn('paintings', 'height')) {
                $table->decimal('height', 8, 2)->nullable()->after('price');
            }
            if (!Schema::hasColumn('paintings', 'width')) {
                $table->decimal('width', 8, 2)->nullable()->after('height');
            }
            if (!Schema::hasColumn('paintings', 'price_on_request')) {
                $table->boolean('price_on_request')->default(false)->after('width');
            }
            if (!Schema::hasColumn('paintings', 'updated_by')) {
                $table->string('updated_by', 100)->nullable()->after('status');
            }
        });

        $this->ensureUniqueIndex('paintings', 'paintings_painting_code_unique', ['painting_code']);
    }

    private function ensureSculpturesTable(): void
    {
        if (!Schema::hasTable('sculptures')) {
            Schema::create('sculptures', function (Blueprint $table) {
                $table->id();
                $table->string('title');
                $table->string('slug')->unique();
                $table->text('description')->nullable();
                $table->decimal('price', 10, 2);
                $table->string('image');
                $table->boolean('status')->default(1);
                $table->timestamps();
            });
        }
    }

    private function ensurePopularSearchesTable(): void
    {
        if (!Schema::hasTable('popular_searches')) {
            Schema::create('popular_searches', function (Blueprint $table) {
                $table->id();
                $table->string('title');
                $table->string('url')->nullable();
                $table->boolean('status')->default(1);
                $table->integer('sort_order')->default(0);
                $table->timestamps();
            });
        }
    }

    private function ensureTestimonialsTable(): void
    {
        if (!Schema::hasTable('testimonials')) {
            Schema::create('testimonials', function (Blueprint $table) {
                $table->id();
                $table->string('name');
                $table->string('photo')->nullable();
                $table->text('message');
                $table->boolean('status')->default(1);
                $table->integer('sort_order')->default(0);
                $table->timestamps();
            });
        }
    }

    private function ensurePageContentsTable(): void
    {
        if (!Schema::hasTable('page_contents')) {
            Schema::create('page_contents', function (Blueprint $table) {
                $table->id();
                $table->string('page_key')->unique();
                $table->longText('content')->nullable();
                $table->boolean('status')->default(1);
                $table->timestamps();
            });
        }
    }

    private function ensureCategoryPaintingTable(): void
    {
        if (!Schema::hasTable('category_painting')) {
            Schema::create('category_painting', function (Blueprint $table) {
                $table->id();
                $table->foreignId('painting_id')->constrained('paintings')->cascadeOnDelete();
                $table->foreignId('category_id')->constrained('categories')->cascadeOnDelete();
                $table->unique(['painting_id', 'category_id']);
            });

            return;
        }

        Schema::table('category_painting', function (Blueprint $table) {
            if (!Schema::hasColumn('category_painting', 'id')) {
                $table->id()->first();
            }
            if (!Schema::hasColumn('category_painting', 'painting_id')) {
                $table->unsignedBigInteger('painting_id')->nullable()->after('id');
            }
            if (!Schema::hasColumn('category_painting', 'category_id')) {
                $table->unsignedBigInteger('category_id')->nullable()->after('painting_id');
            }
        });

        $this->ensureUniqueIndex('category_painting', 'category_painting_painting_id_category_id_unique', ['painting_id', 'category_id']);
    }

    private function repairSeoPagesTable(): void
    {
        if (!Schema::hasTable('seo_pages')) {
            Schema::create('seo_pages', function (Blueprint $table) {
                $table->id();
                $table->string('page_key')->unique();
                $table->string('seo_title', 255);
                $table->string('seo_slug', 255)->unique();
                $table->text('meta_description')->nullable();
                $table->text('meta_keywords')->nullable();
                $table->timestamps();
            });

            return;
        }

        Schema::table('seo_pages', function (Blueprint $table) {
            if (!Schema::hasColumn('seo_pages', 'page_key')) {
                $table->string('page_key')->nullable()->after('id');
            }
            if (!Schema::hasColumn('seo_pages', 'seo_title')) {
                $table->string('seo_title', 255)->nullable()->after('page_key');
            }
            if (!Schema::hasColumn('seo_pages', 'seo_slug')) {
                $table->string('seo_slug', 255)->nullable()->after('seo_title');
            }
            if (!Schema::hasColumn('seo_pages', 'meta_description')) {
                $table->text('meta_description')->nullable()->after('seo_slug');
            }
            if (!Schema::hasColumn('seo_pages', 'meta_keywords')) {
                $table->text('meta_keywords')->nullable()->after('meta_description');
            }
        });

        // Backfill current SEO title column from legacy columns to avoid blank admin/edit values.
        DB::table('seo_pages')
            ->orderBy('id')
            ->get()
            ->each(function ($row) {
                $seoTitle = $row->seo_title ?? null;

                if (blank($seoTitle)) {
                    $fallback = $row->meta_title
                        ?? $row->page_name
                        ?? (!empty($row->page_key) ? Str::headline((string) $row->page_key) : null)
                        ?? 'Page';

                    DB::table('seo_pages')
                        ->where('id', $row->id)
                        ->update(['seo_title' => $fallback]);
                }
            });

        $this->ensureUniqueIndex('seo_pages', 'seo_pages_page_key_unique', ['page_key']);
        $this->ensureUniqueIndex('seo_pages', 'seo_pages_seo_slug_unique', ['seo_slug']);
    }

    private function seedMissingSeoPages(): void
    {
        $defaults = [
            'home' => ['seo_title' => 'Home', 'seo_slug' => ''],
            'about' => ['seo_title' => 'About', 'seo_slug' => 'about'],
            'exclusive' => ['seo_title' => 'Exclusive', 'seo_slug' => 'exhibitions'],
            'artworks' => ['seo_title' => 'Artworks', 'seo_slug' => 'artworks'],
            'artists' => ['seo_title' => 'Artists', 'seo_slug' => 'artists'],
            'blog' => ['seo_title' => 'Blog', 'seo_slug' => 'blogs'],
            'art_advisory' => ['seo_title' => 'Art Advisory', 'seo_slug' => 'art-advisory'],
            'contact' => ['seo_title' => 'Contact', 'seo_slug' => 'contact'],
        ];

        foreach ($defaults as $pageKey => $meta) {
            $existing = DB::table('seo_pages')->where('page_key', $pageKey)->first();

            if (!$existing) {
                DB::table('seo_pages')->insert([
                    'page_key' => $pageKey,
                    'seo_title' => $meta['seo_title'],
                    'seo_slug' => $meta['seo_slug'],
                    'meta_description' => null,
                    'meta_keywords' => null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
                continue;
            }

            $updates = [];
            if (blank($existing->seo_title)) {
                $updates['seo_title'] = $meta['seo_title'];
            }
            if (blank($existing->seo_slug) && $existing->page_key !== 'home') {
                $updates['seo_slug'] = $meta['seo_slug'];
            }

            if (!empty($updates)) {
                $updates['updated_at'] = now();
                DB::table('seo_pages')->where('id', $existing->id)->update($updates);
            }
        }
    }

    private function ensureUniqueIndex(string $table, string $indexName, array $columns): void
    {
        if (!Schema::hasTable($table)) {
            return;
        }

        $hasIndex = DB::table('information_schema.statistics')
            ->where('table_schema', DB::raw('DATABASE()'))
            ->where('table_name', $table)
            ->where('index_name', $indexName)
            ->exists();

        if ($hasIndex) {
            return;
        }

        Schema::table($table, function (Blueprint $tableBlueprint) use ($columns, $indexName) {
            $tableBlueprint->unique($columns, $indexName);
        });
    }
};
