<?php

namespace App\Http\Controllers\Artist;

use App\Http\Controllers\Controller;
use App\Models\Painting;
use App\Models\Category;
use App\Models\Medium;
use App\Models\Form;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class PaintingController extends Controller
{
    /* ================= INDEX ================= */
    public function index()
    {
        $artist = Auth::guard('artist')->user();

        $paintings = Painting::where('artist_id', $artist->id)
            ->with(['form','categories'])
            ->latest()
            ->paginate(10);

        return view('artist.paintings.index', compact('paintings'));
    }

    /* ================= CREATE ================= */
    public function create()
    {
        $categories = Category::all();
        $media = Medium::pluck('name','id');
        $forms = Form::pluck('name','id');

        return view('artist.paintings.create',
            compact('categories','media','forms'));
    }

    /* ================= STORE ================= */
    public function store(Request $request)
    {
        $artist = Auth::guard('artist')->user();

        $data = $request->validate([
            'painting_code' => 'nullable|string|max:50',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'nullable|numeric',
           'medium_id' => 'nullable|exists:mediums,id',
            'form_id' => 'nullable|exists:forms,id',
            'status' => 'nullable|string',
            'updated_by' => 'nullable|string',
            'image' => 'nullable|image|max:10240',
            'category_ids' => 'nullable|array',
        ]);

        // IMAGE
        if ($request->hasFile('image')) {
            $filename = time().'_'.Str::random(6).'.'.$request->image->extension();
            $request->image->move(public_path('uploads/paintings'), $filename);
            $data['image'] = $filename;
        }

        $data['artist_id'] = $artist->id;

        $painting = Painting::create($data);

        // THEMES
        if ($request->category_ids) {
            $painting->categories()->sync($request->category_ids);
        }

        return redirect()
            ->route('artist.paintings.index')
            ->with('success','Artwork created successfully.');
    }

    /* ================= EDIT ================= */
    public function edit(Painting $painting)
    {
        $this->authorizePainting($painting);

        $categories = Category::pluck('name','id');
        $media = Medium::pluck('name','id');
        $forms = Form::pluck('name','id');

        return view('artist.paintings.edit',
            compact('painting','categories','media','forms'));
    }

    /* ================= UPDATE ================= */
    public function update(Request $request, Painting $painting)
    {
        $this->authorizePainting($painting);

        $data = $request->validate([
            'painting_code' => 'nullable|string|max:50',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'nullable|numeric',
           'medium_id' => 'nullable|exists:mediums,id',
            'form_id' => 'nullable|exists:forms,id',
            'status' => 'nullable|string',
            'updated_by' => 'nullable|string',
            'image' => 'nullable|image|max:10240',
            'category_ids' => 'nullable|array',
        ]);

        // IMAGE UPDATE
        if ($request->hasFile('image')) {
            @unlink(public_path('uploads/paintings/'.$painting->image));

            $filename = time().'_'.Str::random(6).'.'.$request->image->extension();
            $request->image->move(public_path('uploads/paintings'), $filename);
            $data['image'] = $filename;
        }

        $painting->update($data);

        // THEMES UPDATE
        $painting->categories()->sync($request->category_ids ?? []);

        return redirect()
            ->route('artist.paintings.index')
            ->with('success','Artwork updated successfully.');
    }

    /* ================= DELETE ================= */
    public function destroy(Painting $painting)
    {
        $this->authorizePainting($painting);

        @unlink(public_path('uploads/paintings/'.$painting->image));
        $painting->categories()->detach();
        $painting->delete();

        return back()->with('success','Artwork deleted.');
    }

    /* ================= SECURITY ================= */
    private function authorizePainting(Painting $painting)
    {
        if ($painting->artist_id !== Auth::guard('artist')->id()) {
            abort(403);
        }
    }
}
