<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Painting;
use App\Models\Artist;
use App\Models\Category;
use App\Models\Gallery;
use App\Models\Medium;
use App\Models\Form;
use App\Imports\ArtworkBulkImport;
use App\Exports\ArtworkBulkTemplateExport;
use App\Exports\ArtworkBulkErrorsExport;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Maatwebsite\Excel\Facades\Excel;
use Maatwebsite\Excel\HeadingRowImport;
use Throwable;

class PaintingController extends Controller
{
    public function index()
    {
        $paintings = Painting::with([
            'artist',
            'categories',
            'gallery',
            'medium',
            'form',
            'artworkType'
        ])->latest()->paginate(10);

        return view('admin.paintings.index', compact('paintings'));
    }

    public function create()
    {
        return view('admin.paintings.create', [
            'categories' => Category::orderBy('name')->get(),
            'artists'    => Artist::pluck('name', 'id'),
            'galleries'  => Gallery::pluck('title', 'id'),
            // Send form_id with each medium so UI filtering does not perform DB lookups in Blade.
            'media'      => Medium::query()->select('id', 'name', 'form_id')->orderBy('name')->get(),
            'forms'      => Form::pluck('name', 'id'),
        ]);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title'             => 'required|string|max:255',
            'painting_code'     => ['required', 'string', 'max:10', 'regex:/^[A-Z0-9]+$/', 'unique:paintings,painting_code'],
            'availability'      => 'nullable|in:available,sold',
            'artist_id'         => 'nullable|exists:artists,id',
            'gallery_id'        => 'nullable|exists:galleries,id',
            'artwork_type_id'   => 'nullable|exists:artwork_types,id',

            'category_ids'      => 'required|array|min:1',
            'category_ids.*'    => 'exists:categories,id',

            'medium_id'         => 'nullable|exists:mediums,id',
            'form_id'           => 'nullable|exists:forms,id',
            'description'       => 'nullable|string',

            'price'             => 'nullable|numeric|min:0',
            'price_on_request'  => 'nullable|boolean',

            'height'            => 'nullable|numeric|min:0',
            'width'             => 'nullable|numeric|min:0',

            'status'            => 'nullable|in:active,inactive,sold',
            'updated_by'        => 'nullable|string|max:100',

            'image'             => 'nullable|image|mimes:jpg,jpeg,png|max:10240',
        ]);

        /* ===== FORM ↔ MEDIUM VALIDATION (OPTION 1 CORE RULE) ===== */
        if (!empty($data['medium_id']) && !empty($data['form_id'])) {
            $medium = Medium::find($data['medium_id']);

            if ($medium && $medium->form_id !== (int) $data['form_id']) {
                return back()
                    ->withInput()
                    ->withErrors([
                        'medium_id' => 'Selected medium does not belong to selected form.'
                    ]);
            }
        }

        /* ===== CATEGORY ===== */
        $categoryIds = $data['category_ids'];
        unset($data['category_ids']);

        $data['slug'] = $data['painting_code'];

        if (($data['availability'] ?? null) === 'sold') {
            $data['status'] = 'sold';
        } elseif (($data['availability'] ?? null) === 'available') {
            $data['status'] = 'active';
        } else {
            $data['status'] = $data['status'] ?? 'active';
        }
        unset($data['availability']);

        /* ===== PRICE LOGIC ===== */
        if (!empty($data['price_on_request']) || $data['status'] === 'sold') {
            $data['price'] = null;
            $data['price_on_request'] = true;
        } else {
            $data['price_on_request'] = false;
        }

        /* ===== UPDATED BY ===== */
        if (empty($data['updated_by']) && Auth::check()) {
            $data['updated_by'] = Auth::user()->name;
        }

        /* ===== IMAGE ===== */
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $data['image'] = time() . '_' . Str::random(6) . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('uploads/paintings'), $data['image']);
        }

        $painting = Painting::create($data);
        $painting->categories()->sync($categoryIds);

        return redirect()
            ->route('admin.paintings.index')
            ->with('success', 'Artwork created successfully.');
    }

    public function edit(Painting $painting)
    {
        $painting->load('categories');

        return view('admin.paintings.edit', [
            'painting'   => $painting,
            'artists'    => Artist::pluck('name', 'id'),
            'categories' => Category::pluck('name', 'id'),
            // Send form_id with each medium so UI filtering does not perform DB lookups in Blade.
            'media'      => Medium::query()->select('id', 'name', 'form_id')->orderBy('name')->get(),
            'forms'      => Form::pluck('name', 'id'),
        ]);
    }

    public function update(Request $request, Painting $painting)
    {
        $data = $request->validate([
            'title'             => 'required|string|max:255',
            'painting_code'     => 'nullable|string|max:100|unique:paintings,painting_code,' . $painting->id,
            'artist_id'         => 'nullable|exists:artists,id',
            'gallery_id'        => 'nullable|exists:galleries,id',
            'artwork_type_id'   => 'nullable|exists:artwork_types,id',

            'category_ids'      => 'nullable|array',
            'category_ids.*'    => 'exists:categories,id',

            'medium_id'         => 'nullable|exists:mediums,id',
            'form_id'           => 'nullable|exists:forms,id',
            'description'       => 'nullable|string',

            'price'             => 'nullable|numeric|min:0',
            'price_on_request'  => 'nullable|boolean',

            'height'            => 'nullable|numeric|min:0',
            'width'             => 'nullable|numeric|min:0',

            'status'            => 'required|in:active,inactive,sold',
            'updated_by'        => 'nullable|string|max:100',

            'image'             => 'nullable|image|mimes:jpg,jpeg,png|max:10240',
        ]);

        /* ===== FORM ↔ MEDIUM VALIDATION (OPTION 1 CORE RULE) ===== */
        if (!empty($data['medium_id']) && !empty($data['form_id'])) {
            $medium = Medium::find($data['medium_id']);

            if ($medium && $medium->form_id !== (int) $data['form_id']) {
                return back()
                    ->withInput()
                    ->withErrors([
                        'medium_id' => 'Selected medium does not belong to selected form.'
                    ]);
            }
        }

        /* ===== CATEGORY ===== */
        $categoryIds = $data['category_ids'] ?? [];
        unset($data['category_ids']);

        if (!empty($data['painting_code'])) {
            $data['slug'] = $data['painting_code'];
        }

        /* ===== PRICE LOGIC ===== */
        if (!empty($data['price_on_request'])) {
            $data['price'] = null;
            $data['price_on_request'] = true;
        } else {
            $data['price_on_request'] = false;
        }

        /* ===== UPDATED BY ===== */
        if (empty($data['updated_by']) && Auth::check()) {
            $data['updated_by'] = Auth::user()->name;
        }

        /* ===== IMAGE ===== */
        if ($request->hasFile('image')) {
            if ($painting->image && file_exists(public_path('uploads/paintings/' . $painting->image))) {
                unlink(public_path('uploads/paintings/' . $painting->image));
            }

            $file = $request->file('image');
            $data['image'] = time() . '_' . Str::random(6) . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('uploads/paintings'), $data['image']);
        }

        $painting->update($data);
        $painting->categories()->sync($categoryIds);

        return redirect()
            ->route('admin.paintings.index')
            ->with('success', 'Artwork updated successfully.');
    }

   public function bulkUpload(Request $request)
{
    $summary = null;
    $rowErrors = [];
    $errorToken = null;

    if ($request->isMethod('post')) {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv|max:20480',
        ]);

        $file = $request->file('file');
        $extension = strtolower($file->getClientOriginalExtension());

        $readerType = match ($extension) {
            'csv'  => \Maatwebsite\Excel\Excel::CSV,
            'xlsx' => \Maatwebsite\Excel\Excel::XLSX,
            default => throw new \Exception('Unsupported file type'),
        };

        $adminName = Auth::guard('admin')->user()?->name
            ?? Auth::user()?->name
            ?? 'Admin';

        $import = new ArtworkBulkImport($adminName);

        try {
            Excel::import($import, $file, null, $readerType);
        } catch (Throwable $e) {
            return back()->withErrors(['file' => $e->getMessage()]);
        }

        $summary = $import->getSummary();
        $rowErrors = $import->getErrors();

        if (!empty($rowErrors)) {
            $errorToken = (string) Str::uuid();
            Cache::put('artwork_bulk_errors_' . $errorToken, $rowErrors, now()->addHour());
        }
    }

    return view('admin.paintings.bulk-upload', compact(
        'summary',
        'rowErrors',
        'errorToken'
    ));
}


    public function downloadBulkTemplate()
    {
        return Excel::download(new ArtworkBulkTemplateExport(), 'artworks_bulk.xlsx');
    }

    public function downloadBulkErrors(string $token)
    {
        $errors = Cache::get('artwork_bulk_errors_' . $token, []);

        abort_if(empty($errors), 404);

        return Excel::download(new ArtworkBulkErrorsExport($errors), 'artworks_bulk_errors.xlsx');
    }

    public function destroy(Painting $painting)
    {
        if ($painting->image && file_exists(public_path('uploads/paintings/' . $painting->image))) {
            unlink(public_path('uploads/paintings/' . $painting->image));
        }

        $painting->delete();

        return redirect()
            ->route('admin.paintings.index')
            ->with('success', 'Artwork deleted.');
    }
}
