<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use App\Models\Exhibition;
use App\Models\ExhibitionImage;

class ExhibitionController extends Controller
{
    /**
     * LIST
     */
    public function index()
    {
        $exhibitions = Exhibition::latest()->paginate(10);
        return view('admin.exhibitions.index', compact('exhibitions'));
    }

    /**
     * CREATE FORM
     */
    public function create()
    {
        return view('admin.exhibitions.create');
    }

    /**
     * STORE
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'title'        => 'required|string|max:255',
            'cover_image'  => 'required|image',
            'event_date'   => 'nullable|date',
            'end_date'     => 'nullable|date|after_or_equal:event_date',
            'event_time'   => 'nullable|string|max:50',
            'venue'        => 'nullable|string|max:255',
            'category'     => 'nullable|string|max:255',
            'description'  => 'nullable|string',
            'gallery.*'    => 'nullable|image',
        ]);

        /* FEATURED */
        $data['is_featured'] = $request->has('is_featured');

        /* UNIQUE SLUG */
        $slug = Str::slug($data['title']);
        $count = Exhibition::where('slug', 'LIKE', "{$slug}%")->count();
        $data['slug'] = $count ? "{$slug}-{$count}" : $slug;

        /* UPDATED BY */
        $data['updated_by'] = Auth::guard('admin')->user()->name ?? 'Admin';

        /* COVER IMAGE */
        $data['cover_image'] = $request
            ->file('cover_image')
            ->store('exhibitions', 'public');

        $exhibition = Exhibition::create($data);

        /* GALLERY */
        if ($request->hasFile('gallery')) {
            foreach ($request->gallery as $img) {
                ExhibitionImage::create([
                    'exhibition_id' => $exhibition->id,
                    'image' => $img->store('exhibitions/gallery', 'public'),
                ]);
            }
        }

        return redirect()
            ->route('admin.exhibitions.index')
            ->with('success', 'Exclusive created successfully');
    }

    /**
     * EDIT FORM
     */
    public function edit(Exhibition $exhibition)
    {
        $exhibition->load('images');
        return view('admin.exhibitions.edit', compact('exhibition'));
    }

    /**
     * UPDATE
     */
    public function update(Request $request, Exhibition $exhibition)
    {
        $data = $request->validate([
            'title'        => 'required|string|max:255',
            'cover_image'  => 'nullable|image',
            'event_date'   => 'nullable|date',
            'end_date'     => 'nullable|date|after_or_equal:event_date',
            'event_time'   => 'nullable|string|max:50',
            'venue'        => 'nullable|string|max:255',
            'category'     => 'nullable|string|max:255',
            'description'  => 'nullable|string',
            'gallery.*'    => 'nullable|image',
        ]);

        /* FEATURED */
        $data['is_featured'] = $request->has('is_featured');

        /* UPDATE SLUG */
        $slug = Str::slug($data['title']);
        $count = Exhibition::where('slug', 'LIKE', "{$slug}%")
            ->where('id', '!=', $exhibition->id)
            ->count();
        $data['slug'] = $count ? "{$slug}-{$count}" : $slug;

        /* UPDATED BY */
        $data['updated_by'] = Auth::guard('admin')->user()->name ?? 'Admin';

        /* COVER IMAGE UPDATE */
        if ($request->hasFile('cover_image')) {
            if ($exhibition->cover_image) {
                Storage::disk('public')->delete($exhibition->cover_image);
            }

            $data['cover_image'] = $request
                ->file('cover_image')
                ->store('exhibitions', 'public');
        }

        $exhibition->update($data);

        /* ADD MORE GALLERY IMAGES */
        if ($request->hasFile('gallery')) {
            foreach ($request->gallery as $img) {
                ExhibitionImage::create([
                    'exhibition_id' => $exhibition->id,
                    'image' => $img->store('exhibitions/gallery', 'public'),
                ]);
            }
        }

        return redirect()
            ->route('admin.exhibitions.index')
            ->with('success', 'Exclusive updated successfully');
    }

    /**
     * DELETE EXHIBITION
     */
    public function destroy(Exhibition $exhibition)
    {
        if ($exhibition->cover_image) {
            Storage::disk('public')->delete($exhibition->cover_image);
        }

        foreach ($exhibition->images as $img) {
            Storage::disk('public')->delete($img->image);
            $img->delete();
        }

        $exhibition->delete();

        return redirect()
            ->route('admin.exhibitions.index')
            ->with('success', 'Exclusive deleted successfully');
    }

    /**
     * DELETE SINGLE GALLERY IMAGE
     */
    public function destroyImage(ExhibitionImage $image)
    {
        Storage::disk('public')->delete($image->image);
        $image->delete();

        return back()->with('success', 'Image removed successfully');
    }
}
