<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Artist;
use App\Imports\ArtistBulkImport;
use App\Exports\ArtistBulkTemplateExport;
use App\Exports\ArtistBulkErrorsExport;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Cache;
use App\Mail\ArtistCredentialsMail;
use Maatwebsite\Excel\Facades\Excel;
use Maatwebsite\Excel\HeadingRowImport;
use Throwable;

class ArtistController extends Controller
{
    /* ================= INDEX ================= */
    public function index()
    {
        $artists = $this->artistIndexQuery()
            ->paginate(10)
            ->withQueryString();

        return view('admin.artists.index', compact('artists'));
    }

    public function featured(Request $request)
    {
        $artists = $this->artistIndexQuery()
            ->where('is_featured', 1)
            ->paginate(10)
            ->withQueryString();

        return view('admin.artists.featured.index', compact('artists'));
    }

    public function bulkUpload(Request $request)
    {
        $summary = null;
        $rowErrors = [];
        $errorToken = null;

        if ($request->isMethod('post')) {
            $request->validate([
                'file' => 'required|file|mimes:xlsx,csv|max:20480',
            ]);

            $file = $request->file('file');
            $extension = strtolower((string) $file?->getClientOriginalExtension());
            $readerType = match ($extension) {
                'csv' => \Maatwebsite\Excel\Excel::CSV,
                'xlsx' => \Maatwebsite\Excel\Excel::XLSX,
                default => throw new \Exception('Unsupported file type'),
            };

            try {
                $headingRows = (new HeadingRowImport())->toArray($file, null, $readerType);
            } catch (Throwable $e) {
                return back()
                    ->withInput()
                    ->withErrors([
                        'file' => 'Import failed: ' . $e->getMessage(),
                    ]);
            }

            $firstHeadingRow = $headingRows[0][0] ?? [];
            $normalizedHeadings = array_map(function ($heading) {
                return Str::of((string) $heading)->lower()->snake()->toString();
            }, $firstHeadingRow);

            $requiredColumns = ['name', 'status'];
            $missingColumns = array_values(array_diff($requiredColumns, $normalizedHeadings));

            if (!empty($missingColumns)) {
                return back()
                    ->withInput()
                    ->withErrors([
                        'file' => 'Missing required columns: ' . implode(', ', $missingColumns),
                    ]);
            }

            $adminName = Auth::guard('admin')->user()?->name
                ?? Auth::user()?->name
                ?? 'Admin';

            $import = new ArtistBulkImport($adminName);

            try {
                Excel::import($import, $file, null, $readerType);
            } catch (Throwable $e) {
                return back()->withErrors(['file' => $e->getMessage()]);
            }

            $summary = $import->getSummary();
            $rowErrors = $import->getErrors();

            if (!empty($rowErrors)) {
                $errorToken = (string) Str::uuid();
                Cache::put('artist_bulk_errors_' . $errorToken, $rowErrors, now()->addHour());
            }
        }

        return view('admin.artists.bulk-upload', [
            'summary' => $summary,
            'rowErrors' => $rowErrors,
            'errorToken' => $errorToken,
        ]);
    }

    public function downloadBulkTemplate()
    {
        return Excel::download(new ArtistBulkTemplateExport(), 'artists_bulk.xlsx');
    }

    public function downloadBulkErrors(string $token)
    {
        $errors = Cache::get('artist_bulk_errors_' . $token, []);

        abort_if(empty($errors), 404);

        return Excel::download(new ArtistBulkErrorsExport($errors), 'artists_bulk_errors.xlsx');
    }

    /* ================= CREATE ================= */
    public function create()
    {
        return view('admin.artists.create');
    }

    /* ================= STORE ================= */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'          => 'required|string|max:255',
            'slug'          => 'nullable|string|max:255|unique:artists,slug',
            'username'      => 'required|string|max:255|unique:artists,username',
            'email'         => 'required|email|unique:artists,email',
            'phone'         => 'nullable|string|max:50',
            'bio'           => 'nullable|string',
            'updated_by'    => 'nullable|string|max:100',
            'is_featured'   => 'nullable|boolean',
            'artwork_image' => 'required|image|max:2048',
            'profile_image' => 'required|image|max:2048',
        ]);

        $slugSource = trim((string) ($data['slug'] ?? ''));
        if ($slugSource === '') {
            $slugSource = $data['name'];
        }
        $baseSlug = $this->normalizeSlug($slugSource);
        if ($baseSlug === '') {
            $baseSlug = 'artist';
        }
        $data['slug'] = $this->makeUniqueSlug($baseSlug);

        $manualUpdatedBy = trim((string) ($data['updated_by'] ?? ''));
        $data['updated_by'] = $manualUpdatedBy !== ''
            ? $manualUpdatedBy
            : (Auth::guard('admin')->user()?->name ?? null);

        if (!empty($data['is_featured'])) {
            $featuredCount = Artist::where('is_featured', true)->count();
            if ($featuredCount >= 10) {
                return back()->withErrors([
                    'is_featured' => 'Only 10 featured artists are allowed.'
                ])->withInput();
            }
        }

        $generatedPassword = Str::random(8);
        $data['password'] = bcrypt($generatedPassword);
        $data['status'] = 'active';

        if ($request->hasFile('artwork_image')) {
            $filename = time() . '_artwork_' . Str::random(6) . '.' . $request->artwork_image->extension();
            $request->artwork_image->move(public_path('uploads/artists'), $filename);
            $data['artwork_image'] = $filename;
        }

        if ($request->hasFile('profile_image')) {
            $filename = time() . '_profile_' . Str::random(6) . '.' . $request->profile_image->extension();
            $request->profile_image->move(public_path('uploads/artists'), $filename);
            $data['profile_image'] = $filename;
        }

        $artist = Artist::create($data);

        Mail::to($artist->email)->send(
            new ArtistCredentialsMail($artist, $generatedPassword)
        );

        return redirect()
            ->route('admin.artists.index')
            ->with('success', 'Artist created & credentials sent successfully.');
    }

    /* ================= EDIT ================= */
    public function edit(Artist $artist)
    {
        return view('admin.artists.edit', compact('artist'));
    }

    /* ================= UPDATE ================= */
    public function update(Request $request, Artist $artist)
    {
        $data = $request->validate([
            'name'          => 'required|string|max:255',
            'slug'          => 'nullable|string|max:255|unique:artists,slug,' . $artist->id,
            'username'      => 'required|string|max:255|unique:artists,username,' . $artist->id,
            'email'         => 'required|email|unique:artists,email,' . $artist->id,
            'phone'         => 'nullable|string|max:50',
            'bio'           => 'nullable|string',
            'updated_by'    => 'nullable|string|max:100',
            'is_featured'   => 'nullable|boolean',
            'artwork_image' => 'nullable|image|max:2048',
            'profile_image' => 'nullable|image|max:2048',
        ]);

        $slugSource = trim((string) ($data['slug'] ?? ''));
        if ($slugSource === '') {
            $slugSource = $data['name'];
        }
        $baseSlug = $this->normalizeSlug($slugSource);
        if ($baseSlug === '') {
            $baseSlug = 'artist';
        }
        $data['slug'] = $this->makeUniqueSlug($baseSlug, $artist->id);

        $manualUpdatedBy = trim((string) ($data['updated_by'] ?? ''));
        $data['updated_by'] = $manualUpdatedBy !== ''
            ? $manualUpdatedBy
            : (Auth::guard('admin')->user()?->name ?? null);

        if (!empty($data['is_featured']) && !$artist->is_featured) {
            $featuredCount = Artist::where('is_featured', true)->count();
            if ($featuredCount >= 10) {
                return back()->withErrors([
                    'is_featured' => 'Only 10 featured artists are allowed.'
                ])->withInput();
            }
        }

        if ($request->hasFile('artwork_image')) {
            @unlink(public_path('uploads/artists/' . $artist->artwork_image));
            $filename = time() . '_artwork_' . Str::random(6) . '.' . $request->artwork_image->extension();
            $request->artwork_image->move(public_path('uploads/artists'), $filename);
            $data['artwork_image'] = $filename;
        }

        if ($request->hasFile('profile_image')) {
            @unlink(public_path('uploads/artists/' . $artist->profile_image));
            $filename = time() . '_profile_' . Str::random(6) . '.' . $request->profile_image->extension();
            $request->profile_image->move(public_path('uploads/artists'), $filename);
            $data['profile_image'] = $filename;
        }

        $artist->update($data);

        return redirect()
            ->route('admin.artists.index')
            ->with('success', 'Artist updated successfully.');
    }

    /* ================= DELETE ================= */
    public function destroy(Artist $artist)
    {
        @unlink(public_path('uploads/artists/' . $artist->artwork_image));
        @unlink(public_path('uploads/artists/' . $artist->profile_image));
        $artist->delete();

        return back()->with('success', 'Artist deleted successfully.');
    }

    /* ================= STATUS ================= */
    public function toggleStatus(Artist $artist)
    {
        $artist->status = $artist->status === 'active' ? 'blocked' : 'active';
        $artist->save();

        return back()->with('success', 'Artist status updated.');
    }

    private function normalizeSlug(?string $value): string
    {
        return Str::of((string) $value)->lower()->slug('-')->value();
    }

    private function artistIndexQuery()
    {
        return Artist::withCount('paintings')
            ->orderByDesc('updated_at');
    }

    private function makeUniqueSlug(string $baseSlug, ?int $ignoreId = null): string
    {
        $slug = $baseSlug;
        $counter = 1;

        while (Artist::where('slug', $slug)
            ->when($ignoreId, fn ($q) => $q->where('id', '!=', $ignoreId))
            ->exists()) {
            $slug = $baseSlug . '-' . $counter;
            $counter++;
        }

        return $slug;
    }
}
